<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\CertificateModel;

class CertificatesController extends BaseController
{
    protected $certificateModel;

    public function __construct()
    {
        $this->certificateModel = new CertificateModel();
    }

    public function index()
    {
        $search = $this->request->getGet('search') ?? '';
        $filter = [
            'penerbit' => $this->request->getGet('penerbit') ?? '',
            'tanggal_dari' => $this->request->getGet('tanggal_dari') ?? '',
            'tanggal_sampai' => $this->request->getGet('tanggal_sampai') ?? ''
        ];
        $sort = $this->request->getGet('sort') ?? 'tanggal_terbit';
        $order = $this->request->getGet('order') ?? 'DESC';
        $page = (int)($this->request->getGet('page') ?? 1);

        $builder = $this->certificateModel->searchAndFilter($search, $filter, $sort, $order);
        $total = $builder->countAllResults(false);
        
        $certificates = $builder->get(10, ($page - 1) * 10)->getResultArray();

        $pager = \Config\Services::pager();
        $pager->store('certificates', $page, 10, $total);

        $data = [
            'title' => 'Kelola Sertifikat',
            'certificates' => $certificates,
            'pager' => $pager,
            'search' => $search,
            'filter' => $filter,
            'sort' => $sort,
            'order' => $order
        ];

        return view('admin/certificates/index', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Sertifikat'
        ];
        return view('admin/certificates/create', $data);
    }

    public function store()
    {
        $rules = [
            'nama_sertifikat' => 'required|min_length[3]|max_length[255]',
            'foto_sertifikat' => 'permit_empty|uploaded[foto_sertifikat]|max_size[foto_sertifikat,5120]|ext_in[foto_sertifikat,jpg,jpeg,png,pdf]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $fileFoto = $this->request->getFile('foto_sertifikat');
        $fotoName = null;

        if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
            $fotoName = $fileFoto->getRandomName();
            $fileFoto->move(WRITEPATH . 'uploads/certificates/', $fotoName);
        }

        $data = [
            'nama_sertifikat' => $this->request->getPost('nama_sertifikat'),
            'penerbit' => $this->request->getPost('penerbit'),
            'tanggal_terbit' => $this->request->getPost('tanggal_terbit'),
            'tanggal_expired' => $this->request->getPost('tanggal_expired'),
            'nomor_sertifikat' => $this->request->getPost('nomor_sertifikat'),
            'link_sertifikat' => $this->request->getPost('link_sertifikat'),
            'foto_sertifikat' => $fotoName,
            'deskripsi' => $this->request->getPost('deskripsi')
        ];

        if ($this->certificateModel->insert($data)) {
            return redirect()->to('admin/certificates')->with('success', 'Sertifikat berhasil ditambahkan!');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal menambahkan sertifikat!');
        }
    }

    public function edit($id)
    {
        $certificate = $this->certificateModel->find($id);
        
        if (!$certificate) {
            return redirect()->to('admin/certificates')->with('error', 'Sertifikat tidak ditemukan!');
        }

        $data = [
            'title' => 'Edit Sertifikat',
            'certificate' => $certificate
        ];
        return view('admin/certificates/edit', $data);
    }

    public function update($id)
    {
        $certificate = $this->certificateModel->find($id);
        
        if (!$certificate) {
            return redirect()->to('admin/certificates')->with('error', 'Sertifikat tidak ditemukan!');
        }

        $rules = [
            'nama_sertifikat' => 'required|min_length[3]|max_length[255]',
            'foto_sertifikat' => 'permit_empty|uploaded[foto_sertifikat]|max_size[foto_sertifikat,5120]|ext_in[foto_sertifikat,jpg,jpeg,png,pdf]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $fileFoto = $this->request->getFile('foto_sertifikat');
        
        $data = [
            'nama_sertifikat' => $this->request->getPost('nama_sertifikat'),
            'penerbit' => $this->request->getPost('penerbit'),
            'tanggal_terbit' => $this->request->getPost('tanggal_terbit'),
            'tanggal_expired' => $this->request->getPost('tanggal_expired'),
            'nomor_sertifikat' => $this->request->getPost('nomor_sertifikat'),
            'link_sertifikat' => $this->request->getPost('link_sertifikat'),
            'deskripsi' => $this->request->getPost('deskripsi')
        ];

        if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
            if ($certificate['foto_sertifikat'] && file_exists(WRITEPATH . 'uploads/certificates/' . $certificate['foto_sertifikat'])) {
                unlink(WRITEPATH . 'uploads/certificates/' . $certificate['foto_sertifikat']);
            }
            $fotoName = $fileFoto->getRandomName();
            $fileFoto->move(WRITEPATH . 'uploads/certificates/', $fotoName);
            $data['foto_sertifikat'] = $fotoName;
        }

        if ($this->certificateModel->update($id, $data)) {
            return redirect()->to('admin/certificates')->with('success', 'Sertifikat berhasil diupdate!');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal mengupdate sertifikat!');
        }
    }

    public function delete($id)
    {
        $certificate = $this->certificateModel->find($id);
        
        if (!$certificate) {
            return redirect()->to('admin/certificates')->with('error', 'Sertifikat tidak ditemukan!');
        }

        if ($certificate['foto_sertifikat'] && file_exists(WRITEPATH . 'uploads/certificates/' . $certificate['foto_sertifikat'])) {
            unlink(WRITEPATH . 'uploads/certificates/' . $certificate['foto_sertifikat']);
        }

        if ($this->certificateModel->delete($id)) {
            return redirect()->to('admin/certificates')->with('success', 'Sertifikat berhasil dihapus!');
        } else {
            return redirect()->to('admin/certificates')->with('error', 'Gagal menghapus sertifikat!');
        }
    }
}

